# Local to Live Configuration & File Sync Guide

## Overview

Your Laravel Crowdfunding/E-Commerce system stores configuration in **two places**:

1. **Database** (`business_settings` table) - Contains all admin-configured settings
2. **File System** (`public/storage/`) - Contains uploaded images/files (logos, banners, products)

To replicate your local setup on live, you need to sync both.

---

## Database Tables to Sync

### Core Configuration Tables (MOST IMPORTANT)

| Table | Purpose | Contents |
|-------|---------|----------|
| **business_settings** | All admin settings | Company name, logo, banner, colors, email, etc. |
| **banners** | Home page banners | Slider images, section banners, popups |
| **brands** | Product brands | Brand data and logos |
| **categories** | Product categories | Category names and icons |
| **shop_settings** | Shop-specific configs | Per-shop customizations |

### Image/File Reference Tables

| Table | Purpose | Contains |
|-------|---------|----------|
| **products** | Product list | Product names, descriptions, image paths |
| **shops** | Vendor shops | Shop logos, banners, images |
| **sellers** | Seller profiles | Profile images, logos |
| **customers** | User profiles | Profile pictures |

---

## File Folders to Copy

### Critical Folders (Must Copy)

```
public/storage/
├── company/              ← Company logos, favicons, footer images
├── banner/               ← Homepage banners and sliders
├── brand/                ← Brand logos
├── category/             ← Category icons/images
├── shop/                 ← Vendor shop logos and banners
├── product/              ← Product images
│   ├── thumbnail/
│   └── [other variants]
├── profile/              ← Customer/seller profile pictures
├── seller/               ← Seller profile images
└── delivery-man/         ← Delivery partner images
```

### Optional Folders (Reference Only)

```
public/assets/           ← Theme CSS/JS (usually in git)
public/themes/           ← Theme files (usually in git)
resources/views/         ← Blade templates (usually in git)
resources/themes/        ← Theme views (usually in git)
```

---

## Step-by-Step Sync Process

### OPTION 1: Database Dump + File Copy (RECOMMENDED)

#### Step 1: Export Local Database

```bash
# On local machine
cd /path/to/crowdfunding

# Backup local database
mysqldump -u root -p multivendor > local_config_backup.sql

# Check file size (should be > 100KB if has data)
ls -lh local_config_backup.sql
```

#### Step 2: Export Only Settings Tables (Recommended)

```bash
# If you only want configuration (not all data)
mysqldump -u root -p multivendor \
  business_settings \
  banners \
  brands \
  categories \
  shop_settings \
  settings \
  > config_only.sql

# Upload to live server
scp config_only.sql username@yourdomain.com:/home/username/public_html/
```

#### Step 3: Import on Live Server

```bash
# SSH into live server
ssh username@yourdomain.com

# Import the database
cd /home/username/public_html
mysql -u live_user -p'password' live_db_name < config_only.sql

# Or if you want to merge (safer):
mysql -u live_user -p'password' live_db_name << EOF
-- This will UPDATE existing records
INSERT IGNORE INTO business_settings SELECT * FROM config_settings;
EOF
```

#### Step 4: Copy Storage Folder

```bash
# On local machine, create tar archive of storage folder
cd /path/to/crowdfunding
tar -czf public_storage.tar.gz public/storage/

# Upload to live
scp public_storage.tar.gz username@yourdomain.com:/home/username/public_html/

# On live server, extract
ssh username@yourdomain.com
cd /home/username/public_html
tar -xzf public_storage.tar.gz

# Fix permissions
chmod -R 755 public/storage/
```

---

### OPTION 2: Direct MySQL Connection + SCP (For Large Files)

#### Export on Local
```bash
# In one command
mysqldump -u root -p multivendor \
  business_settings \
  banners \
  brands \
  categories \
  > settings_dump.sql

# Compress for faster transfer
gzip settings_dump.sql
```

#### Transfer
```bash
# Copy compressed file
scp settings_dump.sql.gz username@yourdomain.com:/tmp/

# Copy storage folder
scp -r public/storage username@yourdomain.com:/tmp/storage/
```

#### Import on Live
```bash
# SSH to live
ssh username@yourdomain.com

# Decompress and import
gunzip -c /tmp/settings_dump.sql.gz | mysql -u live_user -p'password' live_db_name

# Move storage folder
cp -r /tmp/storage/* /home/username/public_html/public/storage/
chmod -R 755 /home/username/public_html/public/storage/

# Clear cache
cd /home/username/public_html
php artisan cache:clear
php artisan config:cache
```

---

### OPTION 3: Full Database Copy (Simplest, if starting fresh)

#### On Local
```bash
# Export entire database
mysqldump -u root -p --all-databases multivendor > full_backup.sql

# Or specific database with all tables
mysqldump -u root -p multivendor > full_db.sql
```

#### On Live
```bash
# Create clean database
mysql -u root -p
CREATE DATABASE live_eafricamall;
EXIT;

# Import full database
mysql -u root -p live_eafricamall < full_db.sql

# Update .env to use this database
nano /home/username/public_html/.env
# Change: DB_DATABASE=live_eafricamall
```

---

## What Each Folder Contains

### `public/storage/company/`
**Purpose**: Company branding  
**Files Inside**:
- `company_web_logo` - Main website logo
- `company_mobile_logo` - Mobile app logo  
- `company_fav_icon` - Browser favicon
- `company_footer_logo` - Footer logo

**Size**: ~500KB - 5MB  
**Critical**: YES - Site won't display properly without these

---

### `public/storage/banner/`
**Purpose**: Homepage banners and sliders  
**Files Inside**:
- Multiple banner images for different sections
- Each banner referenced in `banners` table

**Size**: ~5-50MB (depending on number of banners)  
**Critical**: YES - Homepage won't look right without these

---

### `public/storage/brand/`
**Purpose**: Brand logos for product filters  
**Files Inside**:
- Brand logo images
- Referenced by `brands` table

**Size**: ~1-10MB  
**Critical**: MEDIUM - Nice to have, but not required

---

### `public/storage/category/`
**Purpose**: Category icons/images  
**Files Inside**:
- Category thumbnail images
- Used in category listings

**Size**: ~1-5MB  
**Critical**: MEDIUM - Better UX with images

---

### `public/storage/product/`
**Purpose**: All product images  
**Structure**:
```
product/
├── thumbnail/    ← Small preview images
├── [other variations]
└── [... more product images]
```

**Size**: ~50-500MB+ (depends on product count)  
**Critical**: YES - Products won't display without images

---

### `public/storage/shop/`
**Purpose**: Vendor/shop logos and banners  
**Files Inside**:
- Shop logos
- Shop banners
- Shop profile images

**Size**: ~2-20MB  
**Critical**: YES - Vendor pages incomplete without these

---

### `public/storage/profile/`
**Purpose**: User/customer profile pictures  
**Files Inside**:
- Customer profile photos
- Seller profile images

**Size**: ~5-50MB  
**Critical**: LOW - Can be empty on new setup

---

## Database Schema Important Notes

### business_settings Table Structure
```sql
CREATE TABLE business_settings (
  id INT PRIMARY KEY AUTO_INCREMENT,
  key VARCHAR(255) UNIQUE,
  value TEXT,
  created_at TIMESTAMP,
  updated_at TIMESTAMP
);
```

**Key Settings** (what you configured locally):
- `company_name` - Your business name
- `company_email` - Email address
- `company_phone` - Phone number
- `company_web_logo` - Path to logo
- `company_fav_icon` - Path to favicon
- `primary_color` - Theme primary color
- `secondary_color` - Theme secondary color
- `currency_model` - Currency type
- `colors` - JSON with color palette

---

## Recommended Sync Strategy

### FASTEST (Copy Everything)
```bash
# 1. Export full database
mysqldump -u root -p multivendor > full.sql

# 2. Compress storage folder
tar -czf storage.tar.gz public/storage/

# 3. Transfer both to live
scp full.sql username@yourdomain.com:/tmp/
scp storage.tar.gz username@yourdomain.com:/tmp/

# 4. Import on live
mysql -u user -p db < /tmp/full.sql
cd /home/username/public_html
tar -xzf /tmp/storage.tar.gz
php artisan cache:clear
```

**Time**: ~5-15 minutes  
**Risk**: Low (clean start)  
**Best For**: Fresh setup on live

---

### SAFEST (Settings Only + Files)
```bash
# 1. Export only config tables
mysqldump -u root -p multivendor \
  business_settings \
  banners \
  brands \
  categories \
  > config.sql

# 2. Transfer
scp config.sql username@yourdomain.com:/tmp/

# 3. Import (INSERT IGNORE doesn't overwrite)
mysql -u user -p live_db < /tmp/config.sql

# 4. Copy storage folder
scp -r public/storage username@yourdomain.com:/home/username/public_html/public/

# 5. Clear cache
php artisan cache:clear
```

**Time**: ~3-10 minutes  
**Risk**: Very Low (preserves existing live data)  
**Best For**: Already have live database with some data

---

### SELECTIVE (Pick What You Need)
```bash
# Only company logos and banners (no products/profiles)
mysqldump -u root -p multivendor \
  business_settings \
  banners \
  > minimal_config.sql

# Copy only critical folders
scp -r public/storage/company username@yourdomain.com:/home/username/public_html/public/storage/
scp -r public/storage/banner username@yourdomain.com:/home/username/public_html/public/storage/
scp -r public/storage/brand username@yourdomain.com:/home/username/public_html/public/storage/

# Import minimal config
mysql -u user -p live_db < /tmp/minimal_config.sql
```

**Time**: ~2-5 minutes  
**Risk**: Lowest (minimal changes)  
**Best For**: Just want to copy branding/settings

---

## Folder Size Reference

```
Typical sizes (for planning):
├── company/          ~2MB
├── banner/           ~15MB
├── brand/            ~5MB
├── category/         ~5MB
├── product/          ~200MB+ (scales with products)
├── shop/             ~20MB
├── profile/          ~50MB
└── delivery-man/     ~10MB

TOTAL: ~300MB+ (highly variable)
```

---

## What NOT to Copy

❌ **Don't Copy These Folders** (they're in git or auto-generated):
- `public/assets/` - CSS/JS files
- `public/themes/` - Theme files
- `resources/views/` - Blade templates
- `resources/themes/` - Theme templates
- `node_modules/` - Dependencies
- `vendor/` - Composer dependencies
- `storage/logs/` - Log files
- `bootstrap/cache/` - Cache files

✅ **Only Copy**:
- `public/storage/` - User uploaded files
- Database tables - Settings and config

---

## Verification Checklist

After syncing, verify on live:

```bash
# 1. Check database imported
mysql -u user -p live_db
> SELECT COUNT(*) FROM business_settings;
# Should show: > 10

# 2. Check storage folder exists
ls -la /home/username/public_html/public/storage/
# Should show: company/ banner/ product/ etc.

# 3. Check file permissions
ls -ld /home/username/public_html/public/storage/
# Should show: drwxr-xr-x or drwxrwxr-x

# 4. Check website loads
curl https://yourdomain.com/
# Should show homepage HTML

# 5. Check logo displays
# Visit https://yourdomain.com in browser
# Logo should be visible in header

# 6. Clear cache
php artisan cache:clear
php artisan config:cache
php artisan view:clear
```

---

## Troubleshooting

### Images Not Showing

**Symptom**: Logos, banners load locally but not on live

**Solution**:
```bash
# 1. Check files exist
ls -la /home/username/public_html/public/storage/company/

# 2. Check permissions
chmod -R 755 /home/username/public_html/public/storage/

# 3. Check in browser inspector
# Right-click → Inspect → Network
# Look for 404 errors on image URLs

# 4. Clear browser cache
# Press Ctrl+Shift+Delete in browser
```

---

### Settings Not Applied

**Symptom**: Colors, company name not showing

**Solution**:
```bash
# 1. Verify database import
mysql -u user -p live_db -e "SELECT * FROM business_settings LIMIT 5;"

# 2. Clear Laravel cache
php artisan cache:clear
php artisan config:cache

# 3. Check if business_settings table has data
mysql -u user -p live_db -e "SELECT COUNT(*) FROM business_settings;"
# Should show > 0

# 4. Verify AppServiceProvider can read it
php artisan tinker
> DB::table('business_settings')->count()
# Should show a number > 0
```

---

### File Permission Issues

**Symptom**: "Permission denied" when uploading new files

**Solution**:
```bash
# Set correct permissions
chmod -R 775 /home/username/public_html/public/storage/

# Set owner (if needed)
chown -R www-data:www-data /home/username/public_html/public/storage/

# Or use your username
chown -R username:www-data /home/username/public_html/public/storage/
```

---

## One-Command Summary

**Complete sync in one go**:
```bash
# On local machine
mysqldump -u root -p multivendor business_settings banners brands categories > config.sql && \
tar -czf storage.tar.gz public/storage/ && \
scp config.sql storage.tar.gz username@yourdomain.com:/tmp/

# On live server
cd /home/username/public_html && \
mysql -u live_user -p'password' live_db < /tmp/config.sql && \
tar -xzf /tmp/storage.tar.gz && \
chmod -R 755 public/storage/ && \
php artisan cache:clear && \
php artisan config:cache && \
echo "Sync complete!"
```

---

## When to Use Each Option

| Scenario | Best Option | Reason |
|----------|------------|--------|
| First time deploying to live | OPTION 1 (Full) | Clean fresh start |
| Live already has some data | OPTION 2 (Selective) | Preserves existing data |
| Just want branding/colors | OPTION 3 (Minimal) | Fastest, lowest risk |
| Adding products to live | Copy folders only | Don't reimport product DB |
| Updating banners/logos | Copy storage folders | No DB needed |

---

## Final Tips

1. **Always backup first**
   ```bash
   mysqldump -u live_user -p'password' live_db > live_backup_before_sync.sql
   ```

2. **Test on subdomain first**
   - Create `test.yourdomain.com`
   - Sync there first
   - Test everything
   - Then update main domain

3. **Use compression for large files**
   ```bash
   # Instead of copying 300MB folder
   tar -czf storage.tar.gz public/storage/  # Becomes ~100MB
   gzip config.sql  # Becomes ~1-2MB
   ```

4. **Monitor upload progress**
   ```bash
   # Watch transfer in real-time
   scp -v large_file.gz username@yourdomain.com:/tmp/
   ```

5. **Keep backups**
   ```bash
   # Keep copies locally
   cp public/storage/ backup_storage_$(date +%Y%m%d)/
   mysqldump ... > backup_$(date +%Y%m%d).sql
   ```

---

**Ready to sync? Start with OPTION 1 (Full Database + Storage Folder)**

